# 機能設計書 15-HTTP Bearer認証

## 概要

本ドキュメントはFastAPIフレームワークにおけるHTTP Bearer認証機能の設計仕様を定義する。

### 本機能の処理概要

HTTP Bearer認証機能は、RFC 6750で定義されたBearerトークン認証スキームを実装する。HTTPBearerクラスをDependsまたはSecurity依存関係として使用することで、エンドポイントにBearer認証を適用できる。JWTトークンなどの認証トークンをAuthorizationヘッダーで送信する際に使用される。

**業務上の目的・背景**：Bearer認証は、OAuth2やOpenID Connectで発行されたアクセストークンを使用する際の標準的な方法である。ステートレスな認証を実現し、マイクロサービスアーキテクチャやモバイルアプリケーションで広く使用されている。HTTPBearerクラスにより、トークンの抽出処理を簡潔に実装できる。

**機能の利用シーン**：
- JWTトークンベースの認証
- OAuth2アクセストークンの検証
- API認証
- マイクロサービス間の認証
- モバイルアプリケーションとの連携

**主要な処理内容**：
1. HTTPリクエストからAuthorizationヘッダーを取得
2. Bearerスキームの検証
3. トークン部分を抽出
4. HTTPAuthorizationCredentialsオブジェクトとして返却
5. 認証失敗時のHTTPException発生（auto_error=Trueの場合）

**関連システム・外部連携**：
- OpenAPI: HTTPセキュリティスキームの定義
- Swagger UI: Bearer認証のテスト
- JWT: トークンの生成・検証（ユーザー実装）

**権限による制御**：Bearer認証自体は権限制御を行わない。トークンに含まれるスコープやロールに基づくアクセス制御は開発者が実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | HTTP Bearer認証のテスト。トークン入力フォーム表示 |

## 機能種別

セキュリティ / 認証

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| bearerFormat | str \| None | No | トークンフォーマットの説明（例: "JWT"） | 文字列 |
| scheme_name | str \| None | No | セキュリティスキーム名 | 文字列 |
| description | str \| None | No | セキュリティスキームの説明 | 文字列 |
| auto_error | bool | No | 認証失敗時に自動でエラーを発生させるか（デフォルト: True） | ブール値 |

### 入力データソース

- HTTPリクエストのAuthorizationヘッダー（`Bearer <token>`）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTTPAuthorizationCredentials | Pydantic BaseModel | スキームとクレデンシャルを含むオブジェクト |
| credentials.scheme | str | 認証スキーム（"Bearer"） |
| credentials.credentials | str | トークン文字列 |

### 出力先

- パスオペレーション関数のパラメータ
- OpenAPIスキーマ（セキュリティスキーム定義）

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信
   └─ Authorization: Bearer <token> ヘッダーの確認
2. ヘッダー解析
   └─ get_authorization_scheme_param関数でスキームとクレデンシャルを分離
3. スキーム検証
   └─ "bearer"（大文字小文字区別なし）であることを確認
4. 結果返却
   └─ HTTPAuthorizationCredentialsオブジェクトを生成して返却
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{Authorizationヘッダーあり?}
    B -->|No| C{auto_error?}
    B -->|Yes| D[スキームとクレデンシャルを分離]
    D --> E{スキームはbearer?}
    E -->|No| C
    E -->|Yes| F{クレデンシャルが空?}
    F -->|Yes| C
    F -->|No| G[HTTPAuthorizationCredentials生成]
    G --> H[返却]
    C -->|True| I[HTTPException 401発生]
    C -->|False| J[None返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-15-01 | Bearerスキーム検証 | Authorizationヘッダーのスキームは"bearer"（大文字小文字区別なし）である必要がある | HTTPBearer使用時 |
| BR-15-02 | クレデンシャル必須 | トークン部分は空であってはならない | 認証情報の検証時 |
| BR-15-03 | WWW-Authenticateヘッダー | 認証エラー時はWWW-Authenticate: Bearerヘッダーを含める | HTTPException発生時 |

### 計算ロジック

特になし。単純な文字列分離処理のみ。

## データベース操作仕様

本機能はデータベース操作を直接行わない。トークンの検証は開発者が実装する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HTTP 401 | Unauthorized | Authorizationヘッダーがない | 正しいトークンを送信 |
| HTTP 401 | Unauthorized | スキームがbearerでない | Bearer形式で送信 |
| HTTP 401 | Unauthorized | トークン部分が空 | 有効なトークンを送信 |
| HTTP 403 | Forbidden | 無効なクレデンシャル（スキームが異なる場合） | 正しいスキームで送信 |

### リトライ仕様

本機能では自動リトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- 文字列分離のみの軽量処理
- 各リクエストで認証処理が実行される

## セキュリティ考慮事項

- トークンの検証ロジック（署名検証、有効期限確認）は開発者が実装する必要がある
- HTTPSを使用してトークンの盗聴を防止すること
- トークンの有効期限を適切に設定すること
- リフレッシュトークンの使用を検討すること
- JWTを使用する場合は、PyJWT等のライブラリで適切に検証すること

## 備考

- HTTPAuthorizationCredentialsはHTTPBasicCredentialsと異なり、スキームとクレデンシャルをそのまま保持する
- bearerFormatはOpenAPIドキュメントに反映されるが、実際の検証は行わない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、認証情報を表すデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | http.py | `fastapi/security/http.py` | HTTPAuthorizationCredentialsクラスの定義（29-66行目） |
| 1-2 | models.py | `fastapi/openapi/models.py` | HTTPBearerモデルの定義（353-356行目） |

**読解のコツ**: `HTTPAuthorizationCredentials`はscheme（スキーム名）とcredentials（トークン）をフィールドに持つ。

#### Step 2: HTTPBearerクラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | http.py | `fastapi/security/http.py` | HTTPBearerクラスの定義（224-319行目） |

**主要処理フロー**:
1. **262-300行目**: __init__メソッドでbearerFormatとHTTPBearerModelを設定
2. **302-308行目**: make_not_authenticated_errorメソッドでエラー生成
3. **310-319行目**: __call__メソッドでBearerトークンを抽出

#### Step 3: 認証フローを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | http.py | `fastapi/security/http.py` | __call__メソッド内の検証ロジック（310-319行目） |

**主要処理フロー**:
- **311行目**: Authorizationヘッダーの取得
- **312行目**: get_authorization_scheme_paramでスキームとクレデンシャルを分離
- **313-314行目**: スキームが"bearer"でない場合の処理
- **315-318行目**: クレデンシャルが空の場合の処理
- **319行目**: HTTPAuthorizationCredentialsの返却

### プログラム呼び出し階層図

```
HTTPBearer(bearerFormat, scheme_name, description, auto_error)
    │
    └─ HTTPBase.__init__()（暗黙的）
           │
           └─ __call__(request)
                  │
                  ├─ request.headers.get("Authorization")
                  │
                  ├─ get_authorization_scheme_param()
                  │      └─ スキームとクレデンシャル分離
                  │
                  └─ スキーム検証（"bearer"）
                         │
                         └─ HTTPAuthorizationCredentials(scheme, credentials)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTTPリクエスト          HTTPBearer                HTTPAuthorizationCredentials
Authorization:    ───▶   __call__()          ───▶   {scheme: "Bearer",
Bearer eyJhbGc...           │                        credentials: "eyJhbGc..."}
                            ▼
                   get_authorization_scheme_param()
                   スキームとクレデンシャル分離
                            │
                            ▼
                   スキーム検証（"bearer"）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| http.py | `fastapi/security/http.py` | ソース | HTTPBearerクラスとHTTPAuthorizationCredentialsの定義 |
| base.py | `fastapi/security/base.py` | ソース | SecurityBaseクラスの定義 |
| models.py | `fastapi/openapi/models.py` | ソース | HTTPBearerモデルの定義 |
| utils.py | `fastapi/security/utils.py` | ソース | get_authorization_scheme_param関数 |
| exceptions.py | `fastapi/exceptions.py` | ソース | HTTPException定義 |
